﻿using System.Net;
using System.Text;
using System.IO;
using System;
using System.Diagnostics;
using System.Collections.Specialized;
using System.Threading.Tasks;
using System.Threading;

namespace PerformanceTesting
{
    public class Dialog : IDialog
    {
        private CookieContainer cookieJar;
        private string address;
        private string responseData;

        public Dialog(string address) 
        {
            cookieJar = new CookieContainer();
            this.address = address;
            responseData = string.Empty;
        }

        public IDialog Get(string action, Func<string, bool> isValid, bool skipLogging)
        {
            return Get(action, null, isValid, skipLogging);
        }

        public IDialog Get(string action, NameValueCollection parameters, Func<string, bool> isValid, bool skipLogging)
        {
            SimpleHttpWebRequest request = new SimpleHttpWebRequest(address + action, parameters, cookieJar);            
            //Stopwatch sw = new Stopwatch();         
   
            if (!skipLogging)
                Console.WriteLine("Get Task {0}-{1} Start {2}", Task.CurrentId.Value.ToString(), action, DateTime.Now.ToString("MM/dd/yyyy hh:mm:ss.fff tt"));
            //sw.Start();
            DateTime start = DateTime.Now;
            HttpWebResponse response = request.Get();
            //Thread.Sleep(4000);
            DateTime end = DateTime.Now;

            long duration = (long)(end - start).TotalMilliseconds;            
            //long duration = (long)(response.LastModified - now).TotalMilliseconds;
            //if (duration > sw.ElapsedMilliseconds)
            //    duration = sw.ElapsedMilliseconds;

            if (!skipLogging)
                Console.WriteLine("Get Task {0}-{1} Stop {2} seconds", Task.CurrentId.Value.ToString(), action, duration/1000.0);

            //process data
            ProcessResponse(response, action, duration, parameters, isValid);
            //sw = null;
            return this;
        }

        public IDialog Post(string action, NameValueCollection formParameters, Func<string, bool> isValid, bool skipLogging)
        {
            SimpleHttpWebRequest request = new SimpleHttpWebRequest(address + action, cookieJar);
            DateTime now = DateTime.Now;
            Stopwatch sw = new Stopwatch();
            sw.Start();

            if (!skipLogging)
                Console.WriteLine("Post Thread {0} Start {1}", System.Threading.Thread.CurrentThread.ManagedThreadId, now.ToString());
            HttpWebResponse response = request.Post(formParameters);
            sw.Stop();

            if (!skipLogging)
                Console.WriteLine("Post Thread {0} Stop {1}", System.Threading.Thread.CurrentThread.ManagedThreadId, DateTime.Now.ToString());

            long duration = (long)(response.LastModified - now).TotalMilliseconds;
            if (duration > sw.ElapsedMilliseconds)
                duration = sw.ElapsedMilliseconds;
            //process data
            ProcessResponse(response, action, duration, formParameters, isValid);

            return this;
        }

        private void ProcessResponse(HttpWebResponse response, string action, long duration, NameValueCollection parameters, Func<string, bool> isValid) 
        {
            Encoding encode = System.Text.Encoding.GetEncoding("utf-8");
            // Pipes the stream to a higher level stream reader with the required encoding format. 
            StreamReader readStream = new StreamReader(response.GetResponseStream(), encode);
            Char[] read = new Char[4096];
            // Reads 256 characters at a time.    
            int count = readStream.Read(read, 0, 4096);
            while (count > 0)
            {
                // Dumps the 256 characters on a string.
                String str = new String(read, 0, count);
                responseData += str;
                count = readStream.Read(read, 0, 4096);
            }
            // Releases the resources of the response.
            response.Close();
            // Releases the resources of the Stream.
            readStream.Close();

            string parametersString = parameters.ToUriData() + "\n";

            //log data
            if (isValid(responseData))
                Logger.Log(action, duration, parametersString + cookieJar.GetCookies(new Uri(address))["ASP.NET_SessionId"] + responseData);
            else
                Logger.LogError(action, parametersString + responseData);
        }

        public string Address
        {
            get
            {
                return address;
            }
            private set
            {
                address = value;
            }
        }

        public string ResponseData
        {
            get { return responseData; }
        }
    }
}
